int numWaves = 7;
float[] amplitudes = new float[numWaves];
float[] frequencies = new float[numWaves];
float[] speeds = new float[numWaves];

// Time control
float t = 0;
boolean manualMode = false;
float timeStep = 0.1;

// Boat
Boat boat;
boolean kinematicMode = false;  // Direct wave-following mode
float smoothing = 0.1;          // Lerp factor (0 = no movement, 1 = instant)

// Wave editor
boolean editMode = false;
int selectedWave = 0;           // Which wave to edit (0-6)
int editProperty = 0;           // 0=amplitude, 1=frequency, 2=speed

// Motor controls (continuous key tracking)
boolean motorLeft = false;
boolean motorRight = false;

// Wave Profiles
int currentProfile = 0;
String[] profileNames = {"RANDOM", "CALM", "CHOPPY", "ROGUE (Sneaky)", "STORMY"};

// Splash ripples
ArrayList<Splash> splashes = new ArrayList<Splash>();

void setup() {
  size(800, 600);
  applyProfile(0); // Start with random
  boat = new Boat(width/2, 0);
}

void applyProfile(int p) {
  currentProfile = p;
  // Reset all waves
  for (int i = 0; i < numWaves; i++) {
    amplitudes[i] = 0;
    frequencies[i] = 0.01;
    speeds[i] = -4;
  }
  
  if (p == 0) { // RANDOM
    for (int i = 0; i < numWaves; i++) {
      amplitudes[i] = random(5, 30);
      frequencies[i] = random(0.005, 0.04);
      speeds[i] = random(-5, -2);
    }
  } else if (p == 1) { // CALM
    for (int i = 0; i < numWaves; i++) {
      amplitudes[i] = random(2, 3);
      frequencies[i] = random(0.02, 0.06);
      speeds[i] = random(-3, -1);
    }
  } else if (p == 2) { // CHOPPY
    for (int i = 0; i < numWaves; i++) {
      amplitudes[i] = random(10, 25);
      frequencies[i] = random(0.04, 0.08);
      speeds[i] = random(-6, -3);
    }
  } else if (p == 3) { // ROGUE (The Sneaky One)
    // Interference pattern: two large waves with near-identical frequencies
    // This creates "beats" where they sum into one giant wave periodically
    amplitudes[0] = 35; frequencies[0] = 0.010; speeds[0] = -3.0; 
    amplitudes[1] = 1; frequencies[1] = 0.1; speeds[1] = -3.2; 
    // Add small noise/ripples
    for (int i = 2; i < numWaves; i++) {
      amplitudes[i] = random(2, 10);
      frequencies[i] = random(0.03, 0.1);
      speeds[i] = random(-8, -4);
    }
  } else if (p == 4) { // STORMY
    for (int i = 0; i < numWaves; i++) {
      amplitudes[i] = random(35, 65);
      frequencies[i] = random(0.005, 0.015);
      speeds[i] = random(-4, -2);
    }
  }
}

void draw() {
  background(255);
  
  // Auto-advance time (pause in edit mode)
  if (!manualMode && !editMode) {
    t += timeStep * 0.1; 
  }
  
  // Draw UI
  fill(50);
  textSize(14);
  textAlign(LEFT, TOP);
  
  if (editMode) {
    // Edit mode UI
    fill(200, 50, 50);
    text("=== WAVE EDITOR (paused) ===", 20, 20);
    fill(50);
    text("1-7: select wave | TAB: cycle property | LEFT/RIGHT: adjust | 'e' to exit", 20, 38);
    
    String[] propNames = {"Amplitude", "Frequency", "Speed"};
    text("Wave " + (selectedWave + 1) + " selected:", 20, 60);
    
    for (int i = 0; i < 3; i++) {
      String prefix = (i == editProperty) ? "> " : "  ";
      float val = 0;
      if (i == 0) val = amplitudes[selectedWave];
      else if (i == 1) val = frequencies[selectedWave];
      else val = speeds[selectedWave];
      
      if (i == editProperty) fill(200, 50, 50);
      else fill(100);
      text(prefix + propNames[i] + ": " + nf(val, 0, 4), 30, 80 + i * 18);
    }
    
    // Show all waves summary
    fill(80);
    text("All waves:", 20, 145);
    for (int w = 0; w < numWaves; w++) {
      String marker = (w == selectedWave) ? ">" : " ";
      text(marker + (w+1) + ": A=" + nf(amplitudes[w], 0, 1) + " F=" + nf(frequencies[w], 0, 3) + " S=" + nf(speeds[w], 0, 1), 30, 163 + w * 16);
    }
  } else {
    // Normal UI
    text("Time: " + (manualMode ? "MANUAL" : "AUTO"), 20, 20);
    text("Physics: " + (kinematicMode ? "KINEMATIC (smoothing: " + nf(smoothing, 0, 2) + ")" : "SPRING"), 20, 38);
    text("Profile: [" + currentProfile + "] " + profileNames[currentProfile], 20, 56);
    text("0-4 profiles | 'e' editor | 'm' time | 'k' physics | 'r' reset | A/D motor", 20, 74);
    text("Click to reposition boat", 20, 92);
    if (manualMode) text("RIGHT arrow to step time", 20, 110);
  }


  translate(0, height/2);
  
  // Draw water (filled)
  fill(100, 150, 255, 100);
  noStroke();
  beginShape();
  vertex(0, height/2);
  for (int x = 0; x <= width; x += 4) {
    float y = getWaveY(x);
    vertex(x, y);
  }
  vertex(width, height/2);
  endShape(CLOSE);
  
  // Draw wave line
  noFill();
  stroke(50, 100, 200);
  strokeWeight(2);
  beginShape();
  for (int x = 0; x <= width; x += 4) {
    float y = getWaveY(x);
    vertex(x, y);
  }
  endShape();

  // Apply motor input from held keys
  if (motorLeft) boat.motorInput = -1;
  if (motorRight) boat.motorInput = 1;
  
  // Update and draw boat
  boat.update();
  boat.display();
  
  // Update splashes (remove dead ones)
  for (int i = splashes.size() - 1; i >= 0; i--) {
    splashes.get(i).update();
    if (splashes.get(i).isDead()) {
      splashes.remove(i);
    }
  }
}

// Get wave height at x position
float getWaveY(float x) {
  float y = 0;
  for (int i = 0; i < numWaves; i++) {
    y += sin(x * frequencies[i] + t * speeds[i]) * amplitudes[i];
  }
  
  // Add splash ripple displacement
  for (Splash s : splashes) {
    y += s.getDisplacement(x);
  }
  
  return y;
}

// Get wave slope (dy/dx) at x position
float getWaveSlope(float x) {
  float epsilon = 1.0;
  float y1 = getWaveY(x - epsilon);
  float y2 = getWaveY(x + epsilon);
  return (y2 - y1) / (2.0 * epsilon);
}

void keyPressed() {
  // Toggle edit mode
  if (key == 'e' || key == 'E') {
    editMode = !editMode;
    return;
  }
  
  if (editMode) {
    // EDIT MODE CONTROLS
    
    // Number keys 1-7 select wave
    if (key >= '1' && key <= '7') {
      selectedWave = key - '1';
    }
    
    // Tab cycles through properties
    if (key == TAB) {
      editProperty = (editProperty + 1) % 3;
    }
    
    // Arrow keys adjust selected property
    float delta = 0;
    if (keyCode == RIGHT) delta = 1;
    if (keyCode == LEFT) delta = -1;
    
    if (delta != 0) {
      if (editProperty == 0) {
        // Amplitude: adjust by 2
        amplitudes[selectedWave] = max(0, amplitudes[selectedWave] + delta * 2);
      } else if (editProperty == 1) {
        // Frequency: adjust by 0.002
        frequencies[selectedWave] = max(0.001, frequencies[selectedWave] + delta * 0.002);
      } else {
        // Speed: adjust by 0.5
        speeds[selectedWave] += delta * 0.5;
      }
    }
    
  } else {
    // NORMAL MODE CONTROLS
    
    if (key == 'm' || key == 'M') {
      manualMode = !manualMode;
    }
    if (key == 'k' || key == 'K') {
      kinematicMode = !kinematicMode;
    }
    if (key == 'r' || key == 'R') {
      // Reset boat and regenerate current profile
      boat = new Boat(width/2, -50);
      applyProfile(currentProfile);
      t = 0;
    }
    
    // Switch Profiles with 0-4
    if (key >= '0' && key <= '4') {
      applyProfile(key - '0');
    }
    
    if (manualMode && keyCode == RIGHT) {
      t += timeStep * 0.05;
    }
    // Adjust smoothing with UP/DOWN (only in kinematic mode)
    if (kinematicMode) {
      if (keyCode == UP) {
        smoothing = min(1.0, smoothing + 0.02);
      }
      if (keyCode == DOWN) {
        smoothing = max(0.01, smoothing - 0.02);
      }
    }
    
    // Motor controls (A/D keys)
    if (key == 'a' || key == 'A') motorLeft = true;
    if (key == 'd' || key == 'D') motorRight = true;
  }
}

void keyReleased() {
  // Release motor keys
  if (key == 'a' || key == 'A') motorLeft = false;
  if (key == 'd' || key == 'D') motorRight = false;
}

void mousePressed() {
  boat = new Boat(mouseX, mouseY - height/2);
}

class Boat {
  // Position (center of boat)
  float x, y;
  float vx, vy;
  
  // Rotation
  float angle;
  float angularVel;
  
  // Water entry tracking
  boolean wasInWater = false;
  
  // Dimensions
  float boatWidth = 80;
  float boatHeight = 20;
  float halfWidth;
  
  // Physics constants
  float mass = 1.0;
  float momentOfInertia;  // Rotational inertia
  float gravity = 1.5;
  float maxHorizontalSpeed = 6.0;
  float maxFallSpeed = 10.0;  // Terminal velocity (max downward speed)
  
  // Spring/buoyancy parameters
  float springK = 0.06;        // Spring stiffness (softer)
  float damping = 0.95;        // Linear velocity damping
  float angularDamping = 0.42; // Angular velocity damping
  float waterDamping = 0.80;   // Extra damping when in water
  float maxSpringForce = 2.0;  // Cap spring force to prevent wild bouncing
  float submersionThreshold = 10.0;  // Depth before buoyancy kicks in (allows natural tilt)
  float verticalOffset = 5.0;       // Visual offset: positive: boat sits higher in water, negative = boat sits lower in water
  
  // Motor
  float motorThrust = 4;    // How strong the motor pushes
  float motorInput = 0;       // Current motor input (-1 = reverse, 0 = off, 1 = forward)
  
  Boat(float x, float y) {
    this.x = x;
    this.y = y;
    this.vx = 0;
    this.vy = 0;
    this.angle = 0;
    this.angularVel = 0;
    this.halfWidth = boatWidth / 2;
    
    // Moment of inertia for a rectangle rotating about center
    // I = (1/12) * m * (w^2 + h^2)
    this.momentOfInertia = (mass / 12.0) * (boatWidth * boatWidth + boatHeight * boatHeight);
  }
  
  void update() {
    // Get positions of left, center, and right edge points (bottom of boat)
    PVector leftPoint = getEdgePoint(-halfWidth);
    PVector centerPoint = getEdgePoint(0);
    PVector rightPoint = getEdgePoint(halfWidth);
    
    // Get water heights at each point
    float waterLeft = getWaveY(leftPoint.x);
    float waterCenter = getWaveY(centerPoint.x);
    float waterRight = getWaveY(rightPoint.x);
    
    if (kinematicMode) {
      // KINEMATIC MODE: Directly follow wave surface with smoothing
      
      // Target Y = average of water heights at edges (minus boat height offset)
      float targetY = (waterLeft + waterRight) / 2 - boatHeight/2;
      
      // Target angle = angle of the line between left and right water points
      float targetAngle = atan2(waterRight - waterLeft, boatWidth);
      
      // Lerp toward targets
      y = lerp(y, targetY, smoothing);
      angle = lerpAngle(angle, targetAngle, smoothing);
      
      // Keep x position (no horizontal movement in kinematic mode)
      
    } else {
      // SPRING PHYSICS MODE with 3 spring points
      
      // Displacement includes verticalOffset - this shifts where the boat "floats"
      // Positive verticalOffset = boat thinks it's higher (floats up more)
      // Negative verticalOffset = boat thinks it's lower (sits deeper in water)
      float leftDisplacement = leftPoint.y - waterLeft + verticalOffset;
      float centerDisplacement = centerPoint.y - waterCenter + verticalOffset;
      float rightDisplacement = rightPoint.y - waterRight + verticalOffset;
      
      // Per-point gravity (distributed across 3 points)
      float pointGravity = gravity / 3.0;
      
      float leftForceX = 0, leftForceY = pointGravity;
      float centerForceX = 0, centerForceY = pointGravity;
      float rightForceX = 0, rightForceY = pointGravity;
      
      // Calculate effective submersion (depth minus threshold)
      float leftEffective = leftDisplacement - submersionThreshold;
      float centerEffective = centerDisplacement - submersionThreshold;
      float rightEffective = rightDisplacement - submersionThreshold;
      
      // Add buoyancy (perpendicular to surface normal)
      if (leftEffective > 0) {
        float bMag = -leftEffective * springK * 3;
        bMag = max(bMag, -maxSpringForce); // Buoyancy is negative-y (up)
        float s = getWaveSlope(leftPoint.x);
        float nMag = sqrt(s*s + 1);
        leftForceX += bMag * (-s / nMag); // Horizontal component of normal
        leftForceY += bMag * (1 / nMag);  // Vertical component of normal
      }
      if (centerEffective > 0) {
        float bMag = -centerEffective * springK * 3;
        bMag = max(bMag, -maxSpringForce);
        float s = getWaveSlope(centerPoint.x);
        float nMag = sqrt(s*s + 1);
        centerForceX += bMag * (-s / nMag);
        centerForceY += bMag * (1 / nMag);
      }
      if (rightEffective > 0) {
        float bMag = -rightEffective * springK * 3;
        bMag = max(bMag, -maxSpringForce);
        float s = getWaveSlope(rightPoint.x);
        float nMag = sqrt(s*s + 1);
        rightForceX += bMag * (-s / nMag);
        rightForceY += bMag * (1 / nMag);
      }
      
      // Velocity-proportional drag
      float dragForceY = -vy * 0.15;
      float dragTorque = -angularVel * 0.3;
      
      // Total forces
      float totalForceX = leftForceX + centerForceX + rightForceX;
      float totalForceY = leftForceY + centerForceY + rightForceY;
      
      // Torque from individual point Y forces
      // (Using Y forces for rotation is usually enough for a 2D boat)
      float torqueLeft = -halfWidth * leftForceY;
      float torqueRight = halfWidth * rightForceY;
      float totalTorque = torqueLeft + torqueRight;
      
      // Apply drag when in water
      boolean inWater = (leftDisplacement > 0 || centerDisplacement > 0 || rightDisplacement > 0);
      if (inWater) {
        totalForceY += dragForceY;
        totalTorque += dragTorque;
        // Horizontal water drag
        totalForceX -= vx * 0.1; 
      }
      
      // Apply forces
      vx += totalForceX / mass;
      vy += totalForceY / mass;
      angularVel += totalTorque / momentOfInertia;
      
      // Apply motor thrust (Along boat's local orientation)
      if (motorInput != 0) {
        vx += cos(angle) * motorInput * motorThrust;
        vy += sin(angle) * motorInput * motorThrust;
      }
      
      // Apply damping
      if (inWater) {
        // Base water damping
        vx *= waterDamping;
        vy *= waterDamping;
        angularVel *= waterDamping;
        
        // Extra velocity-proportional damping (high speed = more resistance)
        // This reduces trembling after high-speed impacts
        float speed = sqrt(vx*vx + vy*vy);
        if (speed > 2) {
          float extraDamp = 1.0 - (speed * 0.02);  // More damping at higher speeds
          extraDamp = max(extraDamp, 0.7);         // Don't over-damp
          vx *= extraDamp;
          vy *= extraDamp;
        }
        
        // Extra angular damping when rotating fast (reduces trembling)
        if (abs(angularVel) > 0.02) {
          angularVel *= 0.85;
        }
      } else {
        vx *= damping;
        vy *= damping;
        angularVel *= angularDamping;
      }
      
      // Spawn splash on water entry
      if (inWater && !wasInWater && vy > 1) {
        float splashStrength = min(vy * 2, 10);  // Much stronger!
        splashes.add(new Splash(x, splashStrength));
      }
      
      // Track water state for next frame
      wasInWater = inWater;
      
      // Reset motor input (requires continuous key press)
      motorInput = 0;
      
      // Clamp fall speed (terminal velocity)
      if (vy > maxFallSpeed) vy = maxFallSpeed;

      //needs abs() and sign():
      if (vx < -maxHorizontalSpeed) vx = -maxHorizontalSpeed;
      if (vx > maxHorizontalSpeed) vx = maxHorizontalSpeed;
      
      // Update position
      x += vx;
      y += vy;
      angle += angularVel;
    }
    
    // Keep on screen
    x = constrain(x, halfWidth, width - halfWidth);
  }
  
  // Get world position of an edge point (offset from center along boat's local x-axis)
  PVector getEdgePoint(float localX) {
    float worldX = x + cos(angle) * localX;
    float worldY = y + sin(angle) * localX + boatHeight/2;  // Bottom of boat
    return new PVector(worldX, worldY);
  }
  
  void display() {
    pushMatrix();
    translate(x, y);
    rotate(angle);
    
    // Draw boat body
    fill(139, 90, 43);  // Brown
    stroke(80, 50, 20);
    strokeWeight(2);
    rectMode(CENTER);
    rect(0, 0, boatWidth, boatHeight, 4);
    
    // Draw deck details
    stroke(100, 60, 30);
    strokeWeight(1);
    line(-boatWidth/4, -boatHeight/2, -boatWidth/4, boatHeight/2);
    line(boatWidth/4, -boatHeight/2, boatWidth/4, boatHeight/2);
    
    popMatrix();
    
    // Draw spring attachment points (debug visualization)
    PVector left = getEdgePoint(-halfWidth);
    PVector center = getEdgePoint(0);
    PVector right = getEdgePoint(halfWidth);
    
    // Get water heights
    float waterLeft = getWaveY(left.x);
    float waterCenter = getWaveY(center.x);
    float waterRight = getWaveY(right.x);
    
    // Draw spring lines to water surface
    stroke(255, 100, 100, 150);
    strokeWeight(2);
    line(left.x, left.y, left.x, waterLeft);
    line(right.x, right.y, right.x, waterRight);
    
    // Draw center spring line only in spring mode
    if (!kinematicMode) {
      stroke(100, 255, 100, 150);
      line(center.x, center.y, center.x, waterCenter);
    }
    
    // Draw edge points (red)
    fill(255, 0, 0);
    noStroke();
    ellipse(left.x, left.y, 8, 8);
    ellipse(right.x, right.y, 8, 8);
    
    // Draw center point (green) only in spring mode
    if (!kinematicMode) {
      fill(0, 255, 0);
      ellipse(center.x, center.y, 8, 8);
    }
    
    // Draw water contact points (blue)
    fill(0, 100, 255);
    ellipse(left.x, waterLeft, 8, 8);
    ellipse(right.x, waterRight, 8, 8);
    if (!kinematicMode) {
      ellipse(center.x, waterCenter, 8, 8);
    }
  }
}

// Helper: Lerp angles correctly (handles wraparound)
float lerpAngle(float a, float b, float t) {
  float diff = b - a;
  // Normalize to -PI to PI
  while (diff > PI) diff -= TWO_PI;
  while (diff < -PI) diff += TWO_PI;
  return a + diff * t;
}

// Splash ripple class
class Splash {
  float centerX;         // Center of splash
  float strength;        // Initial amplitude
  float radius;          // Current radius of ripple ring
  float age;             // How old the splash is
  float maxAge = 120;    // Hold for 3 seconds (at 60fps)
  float expandSpeed = 2.5; 
  float ringWidth = 100; // impact zone

  Splash(float x, float strength) {
    this.centerX = x;
    this.strength = strength;
    this.radius = 0;
    this.age = 0;
  }
  
  void update() {
    radius += expandSpeed;
    age++;
  }
  
  boolean isDead() {
    return age > maxAge;
  }
  
  float getDisplacement(float x) {
    float dx = abs(x - centerX);
    
    // Impact "Crater" - local compression/depression at the start
    // This makes the water look pushed down initially
    float craterSize = ringWidth * 1.5;
    float craterFade = max(0, 1.0 - (age / 40.0)); // Crater disappears quickly
    float crater = 0;
    if (dx < craterSize) {
      crater = -strength * craterFade * exp(-sq(dx / (craterSize * 0.5)));
    }
    
    // Traveling Ring Ripple
    float ringDist = abs(dx - radius);
    if (ringDist > ringWidth) return crater; // Return crater even if ring hasn't reached
    
    // Smooth envelope for the ring
    float envelope = exp(-sq(ringDist / (ringWidth * 0.4)));
    
    // Heavy ripple oscillation
    float ripple = cos((dx - radius) * 0.12);
    
    // Overall fade over time
    float ageFade = 1.0 - (age / maxAge);
    
    return crater + (ripple * envelope * strength * ageFade);
  }
}
