int numWaves = 7;
float[] amplitudes = new float[numWaves];
float[] frequencies = new float[numWaves];
float[] speeds = new float[numWaves];

// Time control
float t = 0;
boolean manualMode = false;
float timeStep = 0.1;

// Boat
Boat boat;
boolean kinematicMode = false;  // Direct wave-following mode
float smoothing = 0.1;          // Lerp factor (0 = no movement, 1 = instant)

// Wave editor
boolean editMode = false;
int selectedWave = 0;           // Which wave to edit (0-6)
int editProperty = 0;           // 0=amplitude, 1=frequency, 2=speed

void setup() {
  size(800, 600);
  
  for (int i = 0; i < numWaves; i++) {
    amplitudes[i] = random(10, 40);
    frequencies[i] = random(0.005, 0.03);
    speeds[i] = -4; // negative = wave moves right
  }
  
  boat = new Boat(width/2, 0);
}

void draw() {
  background(255);
  
  // Auto-advance time (pause in edit mode)
  if (!manualMode && !editMode) {
    t += timeStep * 0.1; 
  }
  
  // Draw UI
  fill(50);
  textSize(14);
  textAlign(LEFT, TOP);
  
  if (editMode) {
    // Edit mode UI
    fill(200, 50, 50);
    text("=== WAVE EDITOR (paused) ===", 20, 20);
    fill(50);
    text("1-7: select wave | TAB: cycle property | LEFT/RIGHT: adjust | 'e' to exit", 20, 38);
    
    String[] propNames = {"Amplitude", "Frequency", "Speed"};
    text("Wave " + (selectedWave + 1) + " selected:", 20, 60);
    
    for (int i = 0; i < 3; i++) {
      String prefix = (i == editProperty) ? "> " : "  ";
      float val = 0;
      if (i == 0) val = amplitudes[selectedWave];
      else if (i == 1) val = frequencies[selectedWave];
      else val = speeds[selectedWave];
      
      if (i == editProperty) fill(200, 50, 50);
      else fill(100);
      text(prefix + propNames[i] + ": " + nf(val, 0, 4), 30, 80 + i * 18);
    }
    
    // Show all waves summary
    fill(80);
    text("All waves:", 20, 145);
    for (int w = 0; w < numWaves; w++) {
      String marker = (w == selectedWave) ? ">" : " ";
      text(marker + (w+1) + ": A=" + nf(amplitudes[w], 0, 1) + " F=" + nf(frequencies[w], 0, 3) + " S=" + nf(speeds[w], 0, 1), 30, 163 + w * 16);
    }
  } else {
    // Normal UI
    text("Time: " + (manualMode ? "MANUAL" : "AUTO"), 20, 20);
    text("Physics: " + (kinematicMode ? "KINEMATIC (smoothing: " + nf(smoothing, 0, 2) + ")" : "SPRING"), 20, 38);
    text("'e' wave editor | 'm' time | 'k' physics | 'r' reset", 20, 56);
    text("Click to reposition boat", 20, 74);
    if (manualMode) text("RIGHT arrow to step time", 20, 92);
  }


  translate(0, height/2);
  
  // Draw water (filled)
  fill(100, 150, 255, 100);
  noStroke();
  beginShape();
  vertex(0, height/2);
  for (int x = 0; x <= width; x += 4) {
    float y = getWaveY(x);
    vertex(x, y);
  }
  vertex(width, height/2);
  endShape(CLOSE);
  
  // Draw wave line
  noFill();
  stroke(50, 100, 200);
  strokeWeight(2);
  beginShape();
  for (int x = 0; x <= width; x += 4) {
    float y = getWaveY(x);
    vertex(x, y);
  }
  endShape();

  // Update and draw boat
  boat.update();
  boat.display();
}

// Get wave height at x position
float getWaveY(float x) {
  float y = 0;
  for (int i = 0; i < numWaves; i++) {
    y += sin(x * frequencies[i] + t * speeds[i]) * amplitudes[i];
  }
  return y;
}

void keyPressed() {
  // Toggle edit mode
  if (key == 'e' || key == 'E') {
    editMode = !editMode;
    return;
  }
  
  if (editMode) {
    // EDIT MODE CONTROLS
    
    // Number keys 1-7 select wave
    if (key >= '1' && key <= '7') {
      selectedWave = key - '1';
    }
    
    // Tab cycles through properties
    if (key == TAB) {
      editProperty = (editProperty + 1) % 3;
    }
    
    // Arrow keys adjust selected property
    float delta = 0;
    if (keyCode == RIGHT) delta = 1;
    if (keyCode == LEFT) delta = -1;
    
    if (delta != 0) {
      if (editProperty == 0) {
        // Amplitude: adjust by 2
        amplitudes[selectedWave] = max(0, amplitudes[selectedWave] + delta * 2);
      } else if (editProperty == 1) {
        // Frequency: adjust by 0.002
        frequencies[selectedWave] = max(0.001, frequencies[selectedWave] + delta * 0.002);
      } else {
        // Speed: adjust by 0.5
        speeds[selectedWave] += delta * 0.5;
      }
    }
    
  } else {
    // NORMAL MODE CONTROLS
    
    if (key == 'm' || key == 'M') {
      manualMode = !manualMode;
    }
    if (key == 'k' || key == 'K') {
      kinematicMode = !kinematicMode;
    }
    if (key == 'r' || key == 'R') {
      boat = new Boat(width/2, -50);
    }
    if (manualMode && keyCode == RIGHT) {
      t += timeStep * 0.05;
    }
    // Adjust smoothing with UP/DOWN (only in kinematic mode)
    if (kinematicMode) {
      if (keyCode == UP) {
        smoothing = min(1.0, smoothing + 0.02);
      }
      if (keyCode == DOWN) {
        smoothing = max(0.01, smoothing - 0.02);
      }
    }
  }
}

void mousePressed() {
  boat = new Boat(mouseX, mouseY - height/2);
}

class Boat {
  // Position (center of boat)
  float x, y;
  float vx, vy;
  
  // Rotation
  float angle;
  float angularVel;
  
  // Dimensions
  float boatWidth = 80;
  float boatHeight = 20;
  float halfWidth;
  
  // Physics constants
  float mass = 1.0;
  float momentOfInertia;  // Rotational inertia
  float gravity = 1.5;
  
  // Spring/buoyancy parameters
  float springK = 0.06;        // Spring stiffness (softer)
  float damping = 0.95;        // Linear velocity damping
  float angularDamping = 0.92; // Angular velocity damping
  float waterDamping = 0.80;   // Extra damping when in water
  float maxSpringForce = 2.0;  // Cap spring force to prevent wild bouncing
  float submersionThreshold = 10.0;  // Depth before buoyancy kicks in (allows natural tilt)
  float verticalOffset = 5.0;       // Visual offset: negative = boat sits lower in water
  
  Boat(float x, float y) {
    this.x = x;
    this.y = y;
    this.vx = 0;
    this.vy = 0;
    this.angle = 0;
    this.angularVel = 0;
    this.halfWidth = boatWidth / 2;
    
    // Moment of inertia for a rectangle rotating about center
    // I = (1/12) * m * (w^2 + h^2)
    this.momentOfInertia = (mass / 12.0) * (boatWidth * boatWidth + boatHeight * boatHeight);
  }
  
  void update() {
    // Get positions of left, center, and right edge points (bottom of boat)
    PVector leftPoint = getEdgePoint(-halfWidth);
    PVector centerPoint = getEdgePoint(0);
    PVector rightPoint = getEdgePoint(halfWidth);
    
    // Get water heights at each point
    float waterLeft = getWaveY(leftPoint.x);
    float waterCenter = getWaveY(centerPoint.x);
    float waterRight = getWaveY(rightPoint.x);
    
    if (kinematicMode) {
      // KINEMATIC MODE: Directly follow wave surface with smoothing
      
      // Target Y = average of water heights at edges (minus boat height offset)
      float targetY = (waterLeft + waterRight) / 2 - boatHeight/2;
      
      // Target angle = angle of the line between left and right water points
      float targetAngle = atan2(waterRight - waterLeft, boatWidth);
      
      // Lerp toward targets
      y = lerp(y, targetY, smoothing);
      angle = lerpAngle(angle, targetAngle, smoothing);
      
      // Keep x position (no horizontal movement in kinematic mode)
      
    } else {
      // SPRING PHYSICS MODE with 3 spring points
      
      // Displacement includes verticalOffset - this shifts where the boat "floats"
      // Positive verticalOffset = boat thinks it's higher (floats up more)
      // Negative verticalOffset = boat thinks it's lower (sits deeper in water)
      float leftDisplacement = leftPoint.y - waterLeft + verticalOffset;
      float centerDisplacement = centerPoint.y - waterCenter + verticalOffset;
      float rightDisplacement = rightPoint.y - waterRight + verticalOffset;
      
      // Per-point gravity (distributed across 3 points)
      float pointGravity = gravity / 3.0;
      
      // Net force at each point = buoyancy (up) + gravity (down)
      // Buoyancy only applies when submerged, gravity always applies
      float leftForce = pointGravity;   // Start with gravity pulling down
      float centerForce = pointGravity;
      float rightForce = pointGravity;
      
      // Add buoyancy (upward force) when submerged past threshold
      // Threshold allows some free submersion before buoyancy kicks in
      float leftEffective = leftDisplacement - submersionThreshold;
      float centerEffective = centerDisplacement - submersionThreshold;
      float rightEffective = rightDisplacement - submersionThreshold;
      
      if (leftEffective > 0) {
        float buoyancy = -leftEffective * springK * 3;
        buoyancy = constrain(buoyancy, -maxSpringForce, 0);
        leftForce += buoyancy;
      }
      if (centerEffective > 0) {
        float buoyancy = -centerEffective * springK * 3;
        buoyancy = constrain(buoyancy, -maxSpringForce, 0);
        centerForce += buoyancy;
      }
      if (rightEffective > 0) {
        float buoyancy = -rightEffective * springK * 3;
        buoyancy = constrain(buoyancy, -maxSpringForce, 0);
        rightForce += buoyancy;
      }
      
      // Velocity-proportional drag
      float dragForceY = -vy * 0.15;
      float dragTorque = -angularVel * 0.3;
      
      // Total vertical force (all 3 points)
      float totalForceY = leftForce + centerForce + rightForce;
      
      // Torque from each point (force × moment arm)
      // Positive torque = clockwise (right side down)
      float torqueLeft = -halfWidth * leftForce;    // Left edge: negative arm
      float torqueRight = halfWidth * rightForce;   // Right edge: positive arm
      // Center contributes no torque (arm = 0)
      float totalTorque = torqueLeft + torqueRight;
      
      // Apply drag when in water
      boolean inWater = (leftDisplacement > 0 || centerDisplacement > 0 || rightDisplacement > 0);
      if (inWater) {
        totalForceY += dragForceY;
        totalTorque += dragTorque;
      }
      
      // Apply forces
      vy += totalForceY / mass;
      angularVel += totalTorque / momentOfInertia;
      
      // Apply damping
      if (inWater) {
        vx *= waterDamping;
        vy *= waterDamping;
        angularVel *= waterDamping;
      } else {
        vx *= damping;
        vy *= damping;
        angularVel *= angularDamping;
      }
      
      // Update position
      x += vx;
      y += vy;
      angle += angularVel;
    }
    
    // Keep on screen
    x = constrain(x, halfWidth, width - halfWidth);
  }
  
  // Get world position of an edge point (offset from center along boat's local x-axis)
  PVector getEdgePoint(float localX) {
    float worldX = x + cos(angle) * localX;
    float worldY = y + sin(angle) * localX + boatHeight/2;  // Bottom of boat
    return new PVector(worldX, worldY);
  }
  
  void display() {
    pushMatrix();
    translate(x, y);
    rotate(angle);
    
    // Draw boat body
    fill(139, 90, 43);  // Brown
    stroke(80, 50, 20);
    strokeWeight(2);
    rectMode(CENTER);
    rect(0, 0, boatWidth, boatHeight, 4);
    
    // Draw deck details
    stroke(100, 60, 30);
    strokeWeight(1);
    line(-boatWidth/4, -boatHeight/2, -boatWidth/4, boatHeight/2);
    line(boatWidth/4, -boatHeight/2, boatWidth/4, boatHeight/2);
    
    popMatrix();
    
    // Draw spring attachment points (debug visualization)
    PVector left = getEdgePoint(-halfWidth);
    PVector center = getEdgePoint(0);
    PVector right = getEdgePoint(halfWidth);
    
    // Get water heights
    float waterLeft = getWaveY(left.x);
    float waterCenter = getWaveY(center.x);
    float waterRight = getWaveY(right.x);
    
    // Draw spring lines to water surface
    stroke(255, 100, 100, 150);
    strokeWeight(2);
    line(left.x, left.y, left.x, waterLeft);
    line(right.x, right.y, right.x, waterRight);
    
    // Draw center spring line only in spring mode
    if (!kinematicMode) {
      stroke(100, 255, 100, 150);
      line(center.x, center.y, center.x, waterCenter);
    }
    
    // Draw edge points (red)
    fill(255, 0, 0);
    noStroke();
    ellipse(left.x, left.y, 8, 8);
    ellipse(right.x, right.y, 8, 8);
    
    // Draw center point (green) only in spring mode
    if (!kinematicMode) {
      fill(0, 255, 0);
      ellipse(center.x, center.y, 8, 8);
    }
    
    // Draw water contact points (blue)
    fill(0, 100, 255);
    ellipse(left.x, waterLeft, 8, 8);
    ellipse(right.x, waterRight, 8, 8);
    if (!kinematicMode) {
      ellipse(center.x, waterCenter, 8, 8);
    }
  }
}

// Helper: Lerp angles correctly (handles wraparound)
float lerpAngle(float a, float b, float t) {
  float diff = b - a;
  // Normalize to -PI to PI
  while (diff > PI) diff -= TWO_PI;
  while (diff < -PI) diff += TWO_PI;
  return a + diff * t;
}
