
class Spring1D {
  float position;
  float restingPosition;
  
  float k;
  float damping;
  float mass = 1.0;
  
  float force = 0.0;
  float velocity = 0.0;
  
  // Debug forces
  float kForce = 0.0;
  float dForce = 0.0;
 
  Spring1D(float restingPosition, float k, float damping, float mass) {
    this.restingPosition = restingPosition;
    this.position = restingPosition;
    this.k = k; 
    this.damping = damping;  
    this.mass = mass;
}
  
  public void setPosition(float position, boolean resetForce) {
    this.position = position;
    
    if (resetForce) reset();
  }
  
  public void setRestingPosition(float restingPosition, boolean resetForce) {
    this.restingPosition = restingPosition;
    
    if (resetForce) reset();
  }
  
  
  void reset() {
    force = 0.0;
    velocity = 0.0;
    kForce = 0.0;
    dForce = 0.0;
  }
  
  public void sim(float dt) {
    float displacement = restingPosition - position;
    
    kForce = displacement * k;
    float acceleration = kForce / mass;
    velocity += acceleration * dt;
    
    // Calculate damping force (approximated from the velocity reduction)
    dForce = -velocity * damping * mass;
    velocity += (dForce / mass) * dt;
    
    position += velocity * dt;
  }
  
  /*
  ===========
    DEBUG
  ===========
  */
  
  public void debugDraw(float coord, Axis axis, boolean showMouse, boolean showSpring) {
    if (axis == Axis.X) {
      // Draw Spring
      if (showSpring) {
        drawSpringVisual(0, coord, position, height * .5, 20);
      }

      if (showMouse) {
        if (mousePressed) {
          setPosition(mouseX, true); 
          stroke(255, 0, 0); // Red when interacting
        } else {
          stroke(0, 100, 0); // Dark green normally
        }
        
        line(mouseX, 0, mouseX, height);
        noFill();
        circle(mouseX, height * .5, 2.5);
        line(restingPosition, 0, restingPosition, height);
      }
        
      stroke(100, 0, 0); 
      fill(200, 50, 50);
      circle(position, coord, 10.0);
      
      // Draw Forces
      drawForceArrow(position, coord, kForce, 0, color(0, 0, 255)); // K Force (Blue)
      drawForceArrow(position, coord, dForce, 0, color(0, 200, 0)); // Damping (Green)
      
    } else {
      // Draw Spring
      if (showSpring) {
        drawSpringVisual(coord, 0, width * .5, position, 20);
      }

      if (showMouse) {
        if (mousePressed) {
          setPosition(mouseY, true);
          stroke(255, 0, 0); // Red when interacting
        } else {
          stroke(0, 100, 0); // Dark green normally
        }
        
        line(0, mouseY, width, mouseY);
        noFill();
        circle(width * .5, mouseY, 2.5);
        line(0, restingPosition, width, restingPosition);
      }
      
      fill(100, 0, 0);
      circle(coord, position, 10.0);

      // Draw Forces
      drawForceArrow(coord, position, 0, kForce, color(0, 0, 255));
      drawForceArrow(coord, position, 0, dForce, color(0, 200, 0));

    }
  }
  
  private void drawForceArrow(float x, float y, float fx, float fy, int c) {
    if (abs(fx) < 0.1 && abs(fy) < 0.1) return;
    pushMatrix();
    translate(x, y);
    stroke(c);
    fill(c);
    strokeWeight(2);
    line(0, 0, fx * 0.1, fy * 0.1);
    // simple arrow head
    pushMatrix();
    translate(fx * 0.1, fy * 0.1);
    rotate(atan2(fy, fx));
    float arrowSize = 4;
    line(0, 0, -arrowSize, -arrowSize/2);
    line(0, 0, -arrowSize, arrowSize/2);
    popMatrix();
    strokeWeight(1);
    popMatrix();
  }

  private void drawSpringVisual(float x1, float y1, float x2, float y2, int coils) {
    stroke(150);
    noFill();
    beginShape();
    vertex(x1, y1);
    float dx = x2 - x1;
    float dy = y2 - y1;
    float dist = sqrt(dx*dx + dy*dy);
    float nx = -dy / dist;
    float ny = dx / dist;
    
    for (int i = 0; i <= coils; i++) {
        float t = i / (float)coils;
        float px = x1 + dx * t;
        float py = y1 + dy * t;
        if (i != 0 && i != coils) {
            float offset = (i % 2 == 0 ? 1 : -1) * 10;
            px += nx * offset;
            py += ny * offset;
        }
        vertex(px, py);
    }
    endShape();
  }

}
