class WaveData {
  float frequency; // cycles per length
  float amplitude;
  float speed;     // cycles per second
  
  float[] samples; // accessible by Sim class
  
  WaveData(float f, float a, float s) {
     frequency = f;
     amplitude = a;
     speed = s;
     samples = new float[0]; // will be resized by Sim
  }
  
  // Called by Sim class with target sample count
  public void sim(float t, int sampleCount)
  {
    // Resize if needed
    if (samples.length != sampleCount) {
      samples = new float[sampleCount];
    }
    
    if (sampleCount <= 1) {
      println("ERR: WaveData.sim() - sampleCount must be > 1, got: " + sampleCount);
      return;
    }
    
    for (int i = 0; i < sampleCount; i++)
    {
      float xNorm = i / (float)(sampleCount - 1);
      float angle = TWO_PI * (xNorm * frequency);
      samples[i] = sin(angle + t * speed) * amplitude;
    }
  }

  // Returns an interpolated sample based on a normalized position [0, 1]
  public float getSampleAt(float xNorm) {
    if (samples.length == 0) return 0.0;
    
    float pos = xNorm * (samples.length - 1);
    int i0 = floor(pos);
    int i1 = min(i0 + 1, samples.length - 1);
    float frac = pos - i0;
    
    return lerp(samples[i0], samples[i1], frac);
  }
}


class Sim {
  WaveData[] waves;
  int sampleCount;
  float[] cumulativeSamples; // cached cumulative wave heights
  
  Sim(WaveData[] waves, int sampleCount) {
    this.waves = waves;
    this.sampleCount = sampleCount;
    this.cumulativeSamples = new float[sampleCount];
  }
  
  public void sim(float t) {
    // First, simulate each individual wave
    for (WaveData wave : waves) {
      wave.sim(t, sampleCount);
    }
    
    // Then compute and cache the cumulative samples
    for (int i = 0; i < sampleCount; i++) {
      cumulativeSamples[i] = 0;
      for (WaveData wave : waves) {
        cumulativeSamples[i] += wave.samples[i];
      }
    }
  }
  

  
  // Get cached sample directly by index
  public float getCachedSample(int i) {
    if (i < 0 || i >= cumulativeSamples.length) {
      println("ERR: getCachedSample() - index out of bounds: " + i + " (length: " + cumulativeSamples.length + ")");
      return 0.0;
    }
    return cumulativeSamples[i];
  }
  
  // Get cached sample averaged (divided by wave count)
  public float getCachedSampleAveraged(int i) {
    if (i < 0 || i >= cumulativeSamples.length) {
      println("ERR: getCachedSampleAveraged() - index out of bounds: " + i + " (length: " + cumulativeSamples.length + ")");
      return 0.0;
    }
    return cumulativeSamples[i] / waves.length;
  }

  // Returns an interpolated sample from the cached cumulative samples based on a normalized position [0, 1]
  public float getCachedAt(float xNorm) {
    if (cumulativeSamples.length <= 1) {
      println("ERR: getCachedAt() - cumulativeSamples.length must be > 1, got: " + cumulativeSamples.length);
      return 0.0;
    }
    
    float pos = xNorm * (cumulativeSamples.length - 1);
    int i0 = floor(pos);
    int i1 = min(i0 + 1, cumulativeSamples.length - 1);
    float frac = pos - i0;
    
    return lerp(cumulativeSamples[i0], cumulativeSamples[i1], frac);
  }

  // Returns an averaged interpolated sample from the cached cumulative samples based on a normalized position [0, 1]
  public float getCachedAtAveraged(float xNorm) {
    if (waves.length == 0) return 0.0;
    return getCachedAt(xNorm) / waves.length;
  }
}
